<?php
require_once 'config.php';
require_once 'CycleAPI.php';

$api = new CycleAPI();

$from = $_GET['from'] ?? date('Y-m-d', strtotime('-2 days'));
$to = $_GET['to'] ?? date('Y-m-d');
$selectedAgent = $_GET['agent'] ?? '';

$agents = file_exists("agents.json") ? json_decode(file_get_contents("agents.json"), true) : [];

$transactions = $api->getClientTransactions($from, $to);

$agentMap = [];
foreach ($agents as $a) {
    $agentMap[$a['AgentID']] = $a['Name'];
}

if (!empty($selectedAgent)) {
    $transactions = array_filter($transactions, function ($tx) use ($selectedAgent) {
        return isset($tx['PosID']) && $tx['PosID'] == $selectedAgent;
    });
}

function formatAmount($amount) {
    return 'R' . number_format($amount, 2);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Transaction History</title>
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <script src="https://cdn.tailwindcss.com"></script>
  <style>
    .hover-show-copy { position: relative; }
    .copy-btn {
      display: none;
      position: absolute;
      top: 50%;
      right: 5px;
      transform: translateY(-50%);
      background: #6b21a8;
      color: white;
      padding: 2px 6px;
      font-size: 10px;
      border-radius: 6px;
      cursor: pointer;
    }
    .hover-show-copy:hover .copy-btn { display: inline-block; }
    .toast {
      position: fixed;
      bottom: 30px;
      right: 30px;
      background: #6b21a8;
      color: white;
      padding: 8px 14px;
      border-radius: 8px;
      font-size: 14px;
      animation: fadeout 2s forwards;
    }
    @keyframes fadeout {
      0% {opacity: 1;}
      80% {opacity: 1;}
      100% {opacity: 0; transform: translateY(20px);}
    }
    #fullTxnModal {
      display: none;
      position: fixed;
      inset: 0;
      background: rgba(0,0,0,0.5);
      justify-content: center;
      align-items: center;
      z-index: 50;
    }
    #fullTxnModal .modal-content {
      background: white;
      padding: 20px;
      border-radius: 8px;
      width: 90%;
      max-width: 600px;
    }
  </style>
</head>
<body class="bg-gray-50 text-gray-900">
  <div class="min-h-screen flex flex-col">
    <header class="bg-white shadow px-6 py-4 flex items-center justify-between">
      <h1 class="text-xl font-semibold text-purple-700 flex items-center gap-2">
        <img src="https://img.icons8.com/ios-filled/24/receipt-approved.png" class="h-5"> Transaction History
      </h1>
      <a href="admin.php" class="text-sm text-purple-600 hover:underline">← Back to Dashboard</a>
    </header>

    <div class="bg-white px-6 py-4 shadow-sm border-b flex justify-between items-center">
      <form method="GET" class="flex flex-wrap gap-4 items-center">
        <input type="date" name="from" value="<?= $from ?>" required class="border px-4 py-2 rounded-md text-sm shadow-sm">
        <input type="date" name="to" value="<?= $to ?>" required class="border px-4 py-2 rounded-md text-sm shadow-sm">
        <select name="agent" class="border px-4 py-2 rounded-md text-sm shadow-sm">
          <option value="">All Agents</option>
          <?php foreach ($agents as $agent): ?>
            <option value="<?= $agent['AgentID'] ?>" <?= $selectedAgent == $agent['AgentID'] ? 'selected' : '' ?>>
              <?= htmlspecialchars($agent['Name']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <button type="submit" class="bg-purple-600 text-white px-5 py-2 text-sm rounded-md hover:bg-purple-700 transition">Filter</button>
      </form>
      <div class="flex items-center gap-2">
        <input type="text" id="searchBox" placeholder="Search..." class="border px-4 py-2 rounded-md text-sm shadow-sm">
        <button onclick="exportToCSV()" class="border border-gray-400 text-gray-600 px-4 py-2 text-sm rounded-md hover:bg-gray-100">Export CSV</button>
      </div>
    </div>

    <main class="flex-1 overflow-auto px-6 py-6">
      <div class="bg-white shadow-md rounded-lg overflow-auto">
        <table class="min-w-full table-auto text-sm">
          <thead class="bg-gray-100 text-gray-600 uppercase text-xs">
            <tr>
              <th class="px-4 py-3 text-left">Agent</th>
              <th class="px-4 py-3 text-left">Date</th>
              <th class="px-4 py-3 text-left">Amount</th>
              <th class="px-4 py-3 text-left">Type</th>
              <th class="px-4 py-3 text-left">Status</th>
              <th class="px-4 py-3 text-left">Card</th>
              <th class="px-4 py-3 text-left">RRN</th>
              <th class="px-4 py-3 text-left">Acquirer</th>
              <th class="px-4 py-3 text-left">Device</th>
              <th class="px-4 py-3 text-left">Input</th>
              <th class="px-4 py-3 text-left">Lat / Lng</th>
              <th class="px-4 py-3 text-left">Txn ID</th>
              <th class="px-4 py-3 text-left">Txn Response</th>
            </tr>
          </thead>
          <tbody id="txnBody" class="divide-y divide-gray-200">
            <?php if (empty($transactions)): ?>
              <tr><td colspan="13" class="text-center p-6 text-gray-400 italic">No transactions found for this period.</td></tr>
            <?php else: ?>
              <?php foreach ($transactions as $index => $tx): ?>
                <tr class="hover:bg-gray-50">
                  <td class="px-4 py-2"><?= htmlspecialchars($agentMap[$tx['PosID']] ?? 'Unknown') ?></td>
                  <td class="px-4 py-2 whitespace-nowrap"><?= date('Y-m-d H:i', strtotime($tx['Date'])) ?></td>
                  <td class="px-4 py-2"><?= formatAmount($tx['Amount']) ?></td>
                  <td class="px-4 py-2"><?= htmlspecialchars($tx['Operation'] ?? '-') ?></td>
                  <td class="px-4 py-2"><?= htmlspecialchars($tx['StateDisplay'] ?? '-') ?></td>
                  <td class="px-4 py-2">
                    <?php
                      $card = $tx['Card']['PANMasked'] ?? null;
                      $iin = strtolower($tx['Card']['IIN'] ?? '');
                      $apn = strtolower($tx['EMVData']['APN'] ?? '');
                      if (!$card) {
                          echo '<span class="text-gray-500 italic">Cash</span>';
                      } else {
                          echo htmlspecialchars($card);
                          if (strpos($iin, 'visa') !== false || strpos($apn, 'visa') !== false) {
                              echo '<img src="https://img.icons8.com/color/24/visa.png" class="h-5 ml-2" alt="Visa">';
                          } elseif (strpos($iin, 'mastercard') !== false || strpos($apn, 'mastercard') !== false) {
                              echo '<img src="https://img.icons8.com/color/24/mastercard-logo.png" class="h-5 ml-2" alt="MC">';
                          }
                      }
                    ?>
                  </td>
                  <td class="px-4 py-2 hover-show-copy"><?= htmlspecialchars($tx['RRN'] ?? '-') ?>
                    <?php if (!empty($tx['RRN'])): ?>
                      <span class="copy-btn" onclick="copyText('<?= htmlspecialchars($tx['RRN']) ?>')">Copy</span>
                    <?php endif; ?>
                  </td>
                  <td class="px-4 py-2"><?= htmlspecialchars($tx['AcquirerName'] ?? '-') ?></td>
                  <td class="px-4 py-2"><?= htmlspecialchars($tx['DeviceInfo']['DeviceID'] ?? '-') ?></td>
                  <td class="px-4 py-2"><?= htmlspecialchars($tx['InputType'] ?? '-') ?></td>
                  <td class="px-4 py-2"><?= ($tx['Location']['Latitude'] ?? '-') . ', ' . ($tx['Location']['Longitude'] ?? '-') ?></td>
                  <td class="px-4 py-2 hover-show-copy"><?= htmlspecialchars($tx['ID']) ?>
                    <span class="copy-btn" onclick="copyText('<?= htmlspecialchars($tx['ID']) ?>')">Copy</span>
                  </td>
                  <td class="px-4 py-2">
                    <button onclick="openFullModal(`<?= htmlspecialchars(json_encode($tx, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) ?>`)" class="text-purple-600 text-xs underline">View Response</button>
                  </td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </main>
  </div>

<!-- Full Transaction Modal -->
<div id="fullTxnModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center">
  <div class="modal-content">
    <textarea id="modalTextarea" class="w-full h-80 p-4 border rounded-md text-sm mb-4"></textarea>
    <div class="flex justify-end gap-2">
      <button onclick="copyModalContent()" class="bg-purple-600 text-white px-4 py-2 rounded-md hover:bg-purple-700 text-sm">Copy</button>
      <button onclick="closeModal()" class="bg-gray-300 px-4 py-2 rounded-md hover:bg-gray-400 text-sm">Close</button>
    </div>
  </div>
</div>

<script>
function copyText(text) {
  const tempTextarea = document.createElement('textarea');
  tempTextarea.value = text;
  document.body.appendChild(tempTextarea);
  tempTextarea.select();
  document.execCommand('copy');
  document.body.removeChild(tempTextarea);
  showToast('Copied!');
}

function openFullModal(json) {
  const modal = document.getElementById('fullTxnModal');
  const textarea = document.getElementById('modalTextarea');
  textarea.value = json;
  modal.style.display = 'flex';
}

function copyModalContent() {
  const textarea = document.getElementById('modalTextarea');
  textarea.select();
  document.execCommand('copy');
  showToast('Full transaction copied!');
}

function closeModal() {
  document.getElementById('fullTxnModal').style.display = 'none';
}

function showToast(message) {
  const toast = document.createElement('div');
  toast.className = "toast";
  toast.innerText = message;
  document.body.appendChild(toast);
  setTimeout(() => {
    toast.remove();
  }, 2000);
}

document.getElementById('searchBox').addEventListener('input', function () {
  const filter = this.value.toLowerCase();
  const rows = document.querySelectorAll('#txnBody tr');
  rows.forEach(row => {
    row.style.display = row.innerText.toLowerCase().includes(filter) ? '' : 'none';
  });
});

function exportToCSV() {
  let csv = "Agent,Date,Amount,Type,Status,Card,RRN,Acquirer,Device,Input,Lat/Lng,Txn ID\n";
  document.querySelectorAll('#txnBody tr').forEach(row => {
    const cols = row.querySelectorAll('td');
    if (cols.length >= 12) {
      const data = [
        cols[0].innerText.trim(),
        cols[1].innerText.trim(),
        cols[2].innerText.trim(),
        cols[3].innerText.trim(),
        cols[4].innerText.trim(),
        cols[5].innerText.trim(),
        cols[6].innerText.trim(),
        cols[7].innerText.trim(),
        cols[8].innerText.trim(),
        cols[9].innerText.trim(),
        cols[10].innerText.trim(),
        cols[11].innerText.trim()
      ];
      csv += data.map(x => `"${x.replace(/"/g, '""')}"`).join(",") + "\n";
    }
  });

  const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
  const link = document.createElement('a');
  link.href = URL.createObjectURL(blob);
  link.download = 'transactions.csv';
  link.click();
}
</script>
</body>
</html>
