<?php
require_once '../../core/init.php';
require_once 'sidebar.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../auth/admin/login.php");
    exit;
}

$from = $_GET['from'] ?? date('Y-m-01');
$to = $_GET['to'] ?? date('Y-m-d');
$selectedMerchant = $_GET['merchant'] ?? '';
$export = $_GET['export'] ?? '';

$agents = file_exists('../../storage/agents.json') ? json_decode(file_get_contents('../../storage/agents.json'), true) : [];
$agentMap = [];
foreach ($agents as $a) {
    $agentMap[$a['AgentID']] = $a['Name'];
}

// Fetch settlement logs
$query = "SELECT * FROM settlements WHERE DATE(period_start) BETWEEN ? AND ?";
$params = [$from, $to];

if (!empty($selectedMerchant)) {
    $query .= " AND submerchant_id = ?";
    $params[] = $selectedMerchant;
}

$query .= " ORDER BY period_start DESC";

$stmt = $conn->prepare($query);
if (count($params) == 3) {
    $stmt->bind_param("ssi", $params[0], $params[1], $params[2]);
} else {
    $stmt->bind_param("ss", $params[0], $params[1]);
}
$stmt->execute();
$result = $stmt->get_result();
$logs = $result->fetch_all(MYSQLI_ASSOC);

// Handle CSV export
if ($export === 'csv') {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment;filename=settlement_log.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['Payout Reference', 'Merchant', 'Period Start', 'Period End', 'Volume', 'Fees', 'Net Payout', 'Status', 'Payout Date']);
    foreach ($logs as $log) {
        fputcsv($out, [
            $log['payout_reference'] ?? '-',
            $agentMap[$log['submerchant_id']] ?? 'Unknown',
            $log['period_start'],
            $log['period_end'],
            number_format($log['total_amount'], 2),
            number_format($log['total_fees'], 2),
            number_format($log['net_payout'], 2),
            $log['payout_status'],
            $log['payout_date'] ?? '-'
        ]);
    }
    fclose($out);
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Settlements Log | Swirl</title>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50 text-gray-800">
<div class="flex min-h-screen">

  <!-- Sidebar -->
  <?php require_once 'sidebar.php'; ?>

  <main class="ml-64 flex-1 p-8">
    <div class="flex justify-between items-center mb-6">
      <div>
        <h1 class="text-2xl font-bold text-purple-700">📜 Settlements Log</h1>
        <p class="text-gray-500 text-sm">View and export past settlements</p>
      </div>
      <form method="GET" class="flex gap-2 items-center">
        <input type="date" name="from" value="<?= $from ?>" class="p-2 rounded border text-sm">
        <input type="date" name="to" value="<?= $to ?>" class="p-2 rounded border text-sm">
        <select name="merchant" class="p-2 rounded border text-sm">
          <option value="">All Merchants</option>
          <?php foreach ($agents as $a): ?>
            <option value="<?= $a['AgentID'] ?>" <?= $selectedMerchant == $a['AgentID'] ? 'selected' : '' ?>>
              <?= htmlspecialchars($a['Name']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <button type="submit" class="bg-purple-600 hover:bg-purple-700 text-white px-4 py-2 rounded text-sm">Filter</button>
        <button name="export" value="csv" class="bg-green-600 hover:bg-green-700 text-white px-4 py-2 rounded text-sm">⬇️ Export CSV</button>
      </form>
    </div>

    <div class="bg-white shadow-md rounded-lg overflow-x-auto">
      <table class="min-w-full text-sm">
        <thead class="bg-gray-100 text-gray-600 uppercase text-xs">
          <tr>
            <th class="px-4 py-3 text-left">Payout Ref</th>
            <th class="px-4 py-3 text-left">Merchant</th>
            <th class="px-4 py-3 text-left">Period Start</th>
            <th class="px-4 py-3 text-left">Period End</th>
            <th class="px-4 py-3 text-left">Volume</th>
            <th class="px-4 py-3 text-left">Fees</th>
            <th class="px-4 py-3 text-left">Net Payout</th>
            <th class="px-4 py-3 text-left">Status</th>
            <th class="px-4 py-3 text-left">Payout Date</th>
          </tr>
        </thead>
        <tbody class="divide-y divide-gray-200">
          <?php if (empty($logs)): ?>
            <tr><td colspan="9" class="text-center p-6 text-gray-400 italic">No settlements found for this period.</td></tr>
          <?php else: ?>
            <?php foreach ($logs as $log): ?>
              <tr class="hover:bg-gray-50">
                <td class="px-4 py-2"><?= htmlspecialchars($log['payout_reference'] ?? '-') ?></td>
                <td class="px-4 py-2"><?= htmlspecialchars($agentMap[$log['submerchant_id']] ?? 'Unknown') ?></td>
                <td class="px-4 py-2"><?= date('Y-m-d', strtotime($log['period_start'])) ?></td>
                <td class="px-4 py-2"><?= date('Y-m-d', strtotime($log['period_end'])) ?></td>
                <td class="px-4 py-2">R<?= number_format($log['total_amount'], 2) ?></td>
                <td class="px-4 py-2">R<?= number_format($log['total_fees'], 2) ?></td>
                <td class="px-4 py-2 font-semibold text-green-600">R<?= number_format($log['net_payout'], 2) ?></td>
                <td class="px-4 py-2 <?= $log['payout_status'] === 'Paid' ? 'text-green-600 font-semibold' : 'text-yellow-600 font-semibold' ?>">
                  <?= htmlspecialchars($log['payout_status']) ?>
                </td>
                <td class="px-4 py-2"><?= $log['payout_date'] ? date('Y-m-d H:i', strtotime($log['payout_date'])) : '-' ?></td>
              </tr>
            <?php endforeach; ?>
          <?php endif; ?>
        </tbody>
      </table>
    </div>
  </main>

</div>
</body>
</html>
