<?php
require_once '../../core/init.php';
require_once 'sidebar.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../auth/admin/login.php");
    exit;
}

$api = new CycleAPI();

$from = $_GET['from'] ?? date('Y-m-d', strtotime('-7 days'));
$to = $_GET['to'] ?? date('Y-m-d');
$selectedAgent = $_GET['agent'] ?? '';
$view = $_GET['view'] ?? 'ready'; // 'ready' or 'paid'

$agents = file_exists('../../storage/agents.json') ? json_decode(file_get_contents('../../storage/agents.json'), true) : [];
$agentMap = [];
foreach ($agents as $a) {
    $agentMap[$a['AgentID']] = $a['Name'];
}

$transactions = $api->getClientTransactions($from, $to);

if (!empty($selectedAgent)) {
    $transactions = array_filter($transactions, fn($tx) => ($tx['PosID'] ?? '') == $selectedAgent);
}

// Group transactions
$grouped = [];
foreach ($transactions as $tx) {
    $posId = $tx['PosID'] ?? 'unknown';
    $agentName = $agentMap[$posId] ?? 'Unknown';
    $date = date('Y-m-d', strtotime($tx['Date']));
    $grouped[$posId][$date][] = $tx;
}

// Fetch all existing settlements
$existingSettlements = [];
$result = $conn->query("SELECT * FROM settlements");
while ($row = $result->fetch_assoc()) {
    $existingSettlements[$row['submerchant_id']][$row['period_start']] = $row;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Settlements | Swirl</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50 text-gray-800">
<div class="flex min-h-screen">

    <!-- Sidebar -->
    <?php require_once 'sidebar.php'; ?>

    <main class="ml-64 flex-1 p-8 space-y-6">

        <div class="flex justify-between items-center mb-6">
            <h1 class="text-2xl font-bold text-purple-700">💰 Settlements</h1>
            <form method="GET" class="flex gap-2 items-center">
                <input type="date" name="from" value="<?= $from ?>" class="p-2 rounded border text-sm">
                <input type="date" name="to" value="<?= $to ?>" class="p-2 rounded border text-sm">
                <select name="agent" class="p-2 rounded border text-sm">
                    <option value="">All Agents</option>
                    <?php foreach ($agents as $a): ?>
                        <option value="<?= $a['AgentID'] ?>" <?= $selectedAgent == $a['AgentID'] ? 'selected' : '' ?>>
                            <?= htmlspecialchars($a['Name']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <select name="view" class="p-2 rounded border text-sm">
                    <option value="ready" <?= $view === 'ready' ? 'selected' : '' ?>>Ready to Settle</option>
                    <option value="paid" <?= $view === 'paid' ? 'selected' : '' ?>>Paid</option>
                </select>
                <button type="submit" class="bg-purple-600 text-white px-4 py-2 rounded text-sm">Filter</button>
            </form>
        </div>

        <?php
        $hasRecords = false;
        foreach ($grouped as $agentId => $dates):
            foreach ($dates as $date => $txs):

                $total = $fees = $net = 0;
                $agentName = $agentMap[$agentId] ?? 'Unknown';

                $stmt = $conn->prepare("SELECT fee_percent FROM submerchants WHERE cycle_agent_id = ? LIMIT 1");
                $stmt->bind_param("i", $agentId);
                $stmt->execute();
                $res = $stmt->get_result();
                $feePercent = $res->fetch_assoc()['fee_percent'] ?? 2.5;

                foreach ($txs as $tx) {
                    if (($tx['Substate'] ?? null) == 411) {
                        $amt = $tx['Amount'];
                        $fee = $amt * ($feePercent / 100);
                        $total += $amt;
                        $fees += $fee;
                        $net += ($amt - $fee);
                    }
                }

                if ($total <= 0) continue; // Skip if no successful transactions

                $settlementExists = isset($existingSettlements[$agentId]["$date 00:00:00"]);
                $payoutStatus = $settlementExists ? $existingSettlements[$agentId]["$date 00:00:00"]['payout_status'] : 'Unsettled';

                if (($view === 'ready' && !$settlementExists) || ($view === 'paid' && $settlementExists)):

                    $hasRecords = true;
                    $percent = $total > 0 ? round(($net / $total) * 100) : 0;
                ?>
                    <div class="bg-white rounded-lg shadow mb-6 overflow-hidden">
                        <div class="flex justify-between items-center px-4 py-3 bg-gray-100">
                            <div class="font-semibold"><?= $agentName ?> - <?= $date ?></div>
                            <div class="text-sm">
                                Volume: <?= formatAmount($total) ?> | Fees: <?= formatAmount($fees) ?> | Net: <?= formatAmount($net) ?>
                            </div>
                            <?php if (!$settlementExists): ?>
                                <form method="POST" action="actions/mark_settled.php">
                                    <input type="hidden" name="agent_id" value="<?= $agentId ?>">
                                    <input type="hidden" name="date" value="<?= $date ?>">
                                    <button type="submit" class="text-blue-600 hover:underline ml-4 text-sm">Mark as Settled</button>
                                </form>
                            <?php else: ?>
                                <span class="text-green-600 font-medium ml-4 text-sm">✅ Paid</span>
                            <?php endif; ?>
                        </div>

                        <div class="w-full bg-gray-200 h-2">
                            <div class="h-2 bg-green-500" style="width: <?= $percent ?>%"></div>
                        </div>

                        <table class="w-full text-sm">
                            <thead>
                            <tr class="border-b bg-gray-50">
                                <th class="p-2 text-left">Time</th>
                                <th class="p-2 text-left">Amount</th>
                                <th class="p-2 text-left">Card</th>
                                <th class="p-2 text-left">Transaction ID</th>
                            </tr>
                            </thead>
                            <tbody>
                            <?php foreach ($txs as $tx): ?>
                                <tr class="border-t">
                                    <td class="p-2"><?= date('H:i', strtotime($tx['Date'])) ?></td>
                                    <td class="p-2"><?= formatAmount($tx['Amount']) ?></td>
                                    <td class="p-2"><?= $tx['Card']['PANMasked'] ?? 'Cash' ?></td>
                                    <td class="p-2 text-xs"><?= $tx['ID'] ?></td>
                                </tr>
                            <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php
                endif;
            endforeach;
        endforeach;
        ?>

        <?php if (!$hasRecords): ?>
            <p class="text-center text-gray-400 mt-12 italic">No settlements found for the selected filters.</p>
        <?php endif; ?>

    </main>
</div>
</body>
</html>
