<?php
require_once '../../core/init.php';
require_once 'sidebar.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: ../../auth/admin/login.php");
    exit;
}

// Fetch unsettled settlements
$unsettled = $conn->query("SELECT * FROM settlements WHERE payout_status = 'Unsettled' ORDER BY period_start ASC")->fetch_all(MYSQLI_ASSOC);

$agents = file_exists('../../storage/agents.json') ? json_decode(file_get_contents('../../storage/agents.json'), true) : [];
$agentMap = [];
foreach ($agents as $a) {
    $agentMap[$a['AgentID']] = $a['Name'];
}

// Handle batch generation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['selected_settlements'])) {
    $batchId = 'Batch_' . date('Ymd_His');
    $selected = $_POST['selected_settlements'];

    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename=payout_batch_' . date('Ymd_His') . '.csv');
    $out = fopen('php://output', 'w');
    fputcsv($out, ['Merchant', 'Net Payout', 'Period Start', 'Period End', 'Payout Reference']);

    foreach ($selected as $settlementId) {
        $settlement = $conn->query("SELECT * FROM settlements WHERE id = " . intval($settlementId))->fetch_assoc();
        if ($settlement) {
            $merchantName = $agentMap[$settlement['submerchant_id']] ?? 'Unknown';
            fputcsv($out, [
                $merchantName,
                number_format($settlement['net_payout'], 2),
                $settlement['period_start'],
                $settlement['period_end'],
                $batchId
            ]);

            // Update each settlement to "Batch Created"
            $stmt = $conn->prepare("UPDATE settlements SET payout_status = 'Batch Created', payout_reference = ?, payout_date = NOW() WHERE id = ?");
            $stmt->bind_param("si", $batchId, $settlementId);
            $stmt->execute();
        }
    }

    fclose($out);
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <title>Create Payout Batch | Swirl</title>
  <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gray-50 text-gray-800">
<div class="flex min-h-screen">

  <!-- Sidebar -->
  <?php require_once 'sidebar.php'; ?>

  <main class="ml-64 flex-1 p-8">
    <div class="flex justify-between items-center mb-6">
      <div>
        <h1 class="text-2xl font-bold text-purple-700">💸 Create Payout Batch</h1>
        <p class="text-gray-500 text-sm">Select unsettled settlements and generate a payout batch file.</p>
      </div>
    </div>

    <form method="POST">
      <div class="bg-white rounded shadow-md overflow-auto mb-6">
        <table class="min-w-full text-sm">
          <thead class="bg-gray-100 text-gray-600 uppercase text-xs">
            <tr>
              <th class="px-4 py-3"><input type="checkbox" onclick="toggleSelectAll(this)" /></th>
              <th class="px-4 py-3 text-left">Merchant</th>
              <th class="px-4 py-3 text-left">Period Start</th>
              <th class="px-4 py-3 text-left">Period End</th>
              <th class="px-4 py-3 text-left">Net Payout</th>
            </tr>
          </thead>
          <tbody class="divide-y divide-gray-200">
            <?php if (empty($unsettled)): ?>
              <tr><td colspan="5" class="text-center p-6 text-gray-400 italic">No unsettled payouts found.</td></tr>
            <?php else: ?>
              <?php foreach ($unsettled as $settlement): ?>
                <tr class="hover:bg-gray-50">
                  <td class="px-4 py-2">
                    <input type="checkbox" name="selected_settlements[]" value="<?= $settlement['id'] ?>" />
                  </td>
                  <td class="px-4 py-2"><?= htmlspecialchars($agentMap[$settlement['submerchant_id']] ?? 'Unknown') ?></td>
                  <td class="px-4 py-2"><?= date('Y-m-d', strtotime($settlement['period_start'])) ?></td>
                  <td class="px-4 py-2"><?= date('Y-m-d', strtotime($settlement['period_end'])) ?></td>
                  <td class="px-4 py-2 font-semibold text-green-600">R<?= number_format($settlement['net_payout'], 2) ?></td>
                </tr>
              <?php endforeach; ?>
            <?php endif; ?>
          </tbody>
        </table>
      </div>

      <?php if (!empty($unsettled)): ?>
      <div class="text-right">
        <button type="submit" class="bg-purple-600 hover:bg-purple-700 text-white px-6 py-3 rounded-md text-sm shadow">🚀 Generate Batch File & Mark</button>
      </div>
      <?php endif; ?>
    </form>

  </main>

</div>

<script>
function toggleSelectAll(source) {
  checkboxes = document.getElementsByName('selected_settlements[]');
  for(var i=0, n=checkboxes.length;i<n;i++) {
    checkboxes[i].checked = source.checked;
  }
}
</script>
</body>
</html>
